# -*- coding: utf-8 -*-
"""mastering-pipeline-in-transformers.ipynb

Automatically generated by Colab.

Original file is located at
    https://colab.research.google.com/drive/1B81FBE3vtTwKK1vK-Xk24u2Bl_npio7D

# **Understanding Pipelines for Inference in Transformers**

#### **What are Pipelines for Inference?**  
Pipelines in the Hugging Face `transformers` library provide a high-level API that simplifies using models for inference. These pipelines abstract the complexity of preprocessing, model loading, and post-processing, allowing you to run inference without deep knowledge of the underlying model architecture.  

Hugging Face pipelines support various **tasks**, including:  
- **Natural Language Processing (NLP)** (e.g., text classification, question answering)  
- **Computer Vision** (e.g., image classification, object detection)  
- **Speech Processing** (e.g., speech-to-text, text-to-speech)  
- **Multimodal Applications** (e.g., visual question answering)
"""

pip install transformers

pip install torch diffusers

"""**This is to ignore any unwanted warnings as we might not be using any models in pipelines as we are using different examples for understanding**"""

import logging
from transformers.utils import logging as hf_logging

hf_logging.set_verbosity_error()

import warnings
warnings.filterwarnings("ignore")

"""# **Let's look at some basic examples to get started with working with Pipelines**

## **1- Text Summarization**
**Summarizing long articles into concise summaries.**
"""

from transformers import pipeline

summarizer = pipeline("summarization")
text = """Machine learning is a field of artificial intelligence that uses statistical techniques to give computers the ability to learn from data. The field has seen immense growth in the last decade, with applications in various industries such as healthcare, finance, and automation."""
print(summarizer(text, max_length=30, min_length=20))

"""## **2- Sentiment Analysis**
**Classifying text sentiment as positive, negative, or neutral.**
"""

from transformers import pipeline

classifier = pipeline("sentiment-analysis")
result = classifier("I love how user-friendly this library is!")
print(result)

"""## **3- Named Entity Recognition (NER)**
**Detecting named entities like people, locations, and organizations.**
"""

from transformers import pipeline

ner = pipeline("ner")
text = "Elon Musk is the CEO of Tesla and SpaceX, headquartered in the United States."
print(ner(text))

"""## **4- Automatic Speech Recognition (ASR)**
**Transcribing speech from an audio file.**
"""

from transformers import pipeline

asr = pipeline("automatic-speech-recognition", model="openai/whisper-large-v2")
print(asr("https://huggingface.co/datasets/Narsil/asr_dummy/resolve/main/mlk.flac"))

"""## **5- Machine Translation**
**Translating text from one language to another.**
"""

from transformers import pipeline

translator = pipeline("translation_en_to_fr")
print(translator("Hello, how are you today?"))

"""## **6- Question Answering**
**Extracting answers from a given context.**
"""

from transformers import pipeline

qa = pipeline("question-answering")
context = "Hugging Face is a company based in New York that specializes in NLP models."
question = "Where is Hugging Face located?"
print(qa(question=question, context=context))

"""## **7- Image Classification**
**Classifying images into categories.**
"""

from transformers import pipeline

image_classifier = pipeline("image-classification", model="google/vit-base-patch16-224")
image_url = "https://cdn.pixabay.com/photo/2024/12/31/01/02/costa-rica-9301364_960_720.jpg"
print(image_classifier(image_url))

image_classifier = pipeline("image-classification", model="google/vit-base-patch16-224")
image_url = "https://cdn.pixabay.com/photo/2025/01/13/19/40/horse-9331340_1280.jpg"
print(image_classifier(image_url))

"""## **8- Zero-Shot Text Classification**
**Classifying text into categories that were not part of training.**
"""

from transformers import pipeline

classifier = pipeline("zero-shot-classification")
text = "I recently bought an iPhone and it keeps freezing."
labels = ["electronics", "fashion", "sports"]
print(classifier(text, candidate_labels=labels))

"""## **9- Visual Question Answering (VQA)**
**Answering questions about an image.**
"""

from transformers import pipeline

vqa = pipeline("vqa", model="dandelin/vilt-b32-finetuned-vqa")

result = vqa(
    image="https://huggingface.co/datasets/huggingface/documentation-images/resolve/main/pipeline-cat-chonk.jpeg",
    question="What is the animal in the image?"
)
print(result)

"""## **10- Text-to-Speech (TTS) Pipeline**
**Text-to-Speech (TTS) Pipeline**
"""

from transformers import pipeline
import IPython.display as ipd

# Load a compatible text-to-speech model
tts = pipeline("text-to-speech", model="facebook/mms-tts-eng")

# Generate speech
speech = tts("Hugging Face provides powerful AI models! Hi my name is Code Cavalier")

# Play the audio
ipd.Audio(speech["audio"], rate=22050)  # Adjust rate if needed

"""## **11- Fill-in-the-Blank (Masked Language Modeling) Pipeline**
**Predict the missing word in a sentence using BERT-based models.**
"""

mlm = pipeline("fill-mask", model="bert-base-uncased")

result = mlm("Hugging Face is making AI [MASK] for everyone!")
print(result)

"""## **12- Keyword Extraction (Feature Extraction) Pipeline**
**Extract meaningful numerical embeddings for text.**
"""

feature_extractor = pipeline("feature-extraction", model="bert-base-uncased")

embedding = feature_extractor("Hugging Face makes AI accessible.")
print(len(embedding[0]))  # Output: 768 (dimension of BERT embeddings)

"""## **13- Custom Model Loading with Pipeline**
**Use your own fine-tuned Hugging Face model.**
"""

from transformers import AutoModelForSequenceClassification, AutoTokenizer

model_name = "distilbert-base-uncased-finetuned-sst-2-english"  # Replace with your own model
model = AutoModelForSequenceClassification.from_pretrained(model_name)
tokenizer = AutoTokenizer.from_pretrained(model_name)

custom_pipeline = pipeline("text-classification", model=model, tokenizer=tokenizer)
print(custom_pipeline("I love this product!"))

"""## **14- Multimodal AI: Image + Text Analysis**
**Use CLIP to find the best matching caption for an image.**
"""

clip = pipeline("zero-shot-image-classification", model="openai/clip-vit-base-patch16")

result = clip(
    "https://huggingface.co/datasets/huggingface/documentation-images/resolve/main/pipeline-cat-chonk.jpeg",
    candidate_labels=["a fluffy cat", "a dog", "a bird"]
)
print(result)

"""## **15- Custom Tokenizer with Pipeline**
**Use a custom tokenizer for better control over text processing.**
"""

from transformers import AutoTokenizer

tokenizer = AutoTokenizer.from_pretrained("bert-base-uncased")
tokens = tokenizer("Hugging Face is democratizing AI!", return_tensors="pt")
print(tokens)





